/*
 * @(#)OMAssociation.java  1.0  4. Februar 2004
 *
 * Copyright (c) 2003 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */

package ch.hslu.cm.oo.objectmodel;

import java.io.IOException;
import org.jhotdraw.xml.DOMInput;
import org.jhotdraw.xml.DOMOutput;
import java.beans.*;
import ch.hslu.cm.simulation.*;
/**
 * Simulates the structure and behavior of an association between two classes.
 * 
 * @author Werner Randelshofer
 * @version 1.0 4. Februar 2004  Created.
 */
public class OMAssociation extends AbstractRelationship {
    public final static int UNSPECIFIED_MULTIPLICITY = 0;
    public final static int MANY_MULTIPLICITY = -1;
    
    private PropertyChangeSupport changeSupport;
    /**
     * 0 = unspecified
     * -1 = many
     * 1..9999 = count
     */
    private int ownerMultiplicity = 1;
    private int memberMultiplicity = 1;
    private String ownerLabel = "owner";
    private String memberLabel = "member";
    private boolean isMemberTraversable = true;
    private boolean isOwnerTraversable = false;
    
    /** Creates a new instance. */
    public OMAssociation() {
        changeSupport = new PropertyChangeSupport(this);
    }
    
    public void addPropertyChangeListener(PropertyChangeListener l) {
        changeSupport.addPropertyChangeListener(l);
    }
    public void removePropertyChangeListener(PropertyChangeListener l) {
        changeSupport.removePropertyChangeListener(l);
    }
    
    public OMClass getOwner() {
        return (OMClass) getStart();
    }
    public OMClass getMember() {
        return (OMClass) getEnd();
    }
    
    public void setOwnerMultiplicity(int newValue) {
        int oldValue = ownerMultiplicity;
        ownerMultiplicity = newValue;
        changeSupport.firePropertyChange("ownerMultiplicity", oldValue, newValue);
    }
    public int getOwnerMultiplicity() {
        return ownerMultiplicity;
    }
    
    public void setMemberMultiplicity(int newValue) {
        int oldValue = memberMultiplicity;
        memberMultiplicity = newValue;
        changeSupport.firePropertyChange("memberMultiplicity", oldValue, newValue);
    }
    public int getMemberMultiplicity() {
        return memberMultiplicity;
    }
    
    public void setMemberLabel(String newValue) {
        String oldValue = memberLabel;
        memberLabel = newValue;
        changeSupport.firePropertyChange("memberLabel", oldValue, newValue);
    }
    public String getMemberLabel() {
        return memberLabel;
    }
    
    public void setOwnerLabel(String newValue) {
        String oldValue = ownerLabel;
        ownerLabel = newValue;
        changeSupport.firePropertyChange("ownerLabel", oldValue, newValue);
    }
    public String getOwnerLabel() {
        return ownerLabel;
    }
    
    public void setMemberTraversable(boolean newValue) {
        boolean oldValue = isMemberTraversable;
        isMemberTraversable = newValue;
        changeSupport.firePropertyChange("memberTraversable", oldValue, newValue);
    }
    public boolean isMemberTraversable() {
        return isMemberTraversable;
    }
    
    public void setOwnerTraversable(boolean newValue) {
        boolean oldValue = isOwnerTraversable;
        isOwnerTraversable = newValue;
        changeSupport.firePropertyChange("memberTraversable", oldValue, newValue);
    }
    public boolean isOwnerTraversable() {
        return isOwnerTraversable;
    }
    
    public int getSimulatedConcept() {
        return ObjectModel.ASSOCIATION;
    }
    
    public boolean canConnect(SimulatedElement start, SimulatedElement end) {
        if (
        start.getSimulatedConcept() == ObjectModel.CLASS
        && end.getSimulatedConcept() == ObjectModel.CLASS
        ) {
            return true;
        }
        return false;
    }
    public boolean canConnect(SimulatedElement start) {
        return start.getSimulatedConcept() == ObjectModel.CLASS;
    }
    
    public OMAssociation clone() {
        OMAssociation that = (OMAssociation) super.clone();
        that.changeSupport = new PropertyChangeSupport(that);
        return that;
    }
    
    public void read(DOMInput in) throws IOException {
        in.openElement("start");
            ownerMultiplicity = in.getAttribute("multiplicity", UNSPECIFIED_MULTIPLICITY);
        ownerLabel = in.getAttribute("name", "owner");
        isOwnerTraversable = in.getAttribute("traversable", false);
        setStart((OMClass) in.readObject());
        in.closeElement();
        
        in.openElement("end");
            memberMultiplicity = in.getAttribute("multiplicity", UNSPECIFIED_MULTIPLICITY);
        memberLabel = in.getAttribute("name", "member");
        isMemberTraversable = in.getAttribute("traversable", false);
        setEnd((OMClass) in.readObject());
        in.closeElement();
    }
    public void write(DOMOutput out) throws IOException {
        out.openElement("start");
        if (ownerMultiplicity != UNSPECIFIED_MULTIPLICITY) {
            out.addAttribute("multiplicity", ownerMultiplicity);
        }
        out.addAttribute("name", ownerLabel);
        if (isOwnerTraversable) {
            out.addAttribute("traversable", isOwnerTraversable);
        }
        out.writeObject(getStart());
        out.closeElement();
        
        out.openElement("end");
        if (memberMultiplicity != UNSPECIFIED_MULTIPLICITY) {
            out.addAttribute("multiplicity", memberMultiplicity);
        }
        out.addAttribute("name", memberLabel);
        if (isMemberTraversable) {
            out.addAttribute("traversable", isMemberTraversable);
        }
        out.writeObject(getEnd());
        out.closeElement();
    }
    /*
    public String toString() {
        return "OMAssociation ["+ownerLabel+","+isOwnerTraversable+","+ownerMultiplicity+","+getOwner()+"] ["+memberLabel+","+isMemberTraversable+","+memberMultiplicity+","+getMember()+"]";
    }*/
}
